<?php
if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

/**
 * e-bdpay gateway module for WHMCS
 * - Auto-creates invoice if none exists using localAPI CreateInvoice
 * - Builds payment request to bdpay and returns a payment form
 */

function bdpay_MetaData() {
    return array(
        'DisplayName' => 'e-bdpay',
        'APIVersion' => '1.0',
        'DisableLocalCredtCardInput' => true,
        'TokenisedStorage' => false,
    );
}

function e-bdpay_config() {
    return array(
        'FriendlyName' => array(
            'Type' => 'System',
            'Value' => 'e-bdpay',
        ),
        'apiKey' => array(
            'FriendlyName' => 'API Key',
            'Type' => 'text',
            'Size' => '150',
            'Default' => '',
            'Description' => 'Enter Your Api Key',
        ),
        'currency_rate' => array(
            'FriendlyName' => 'Currency Rate',
            'Type' => 'text',
            'Size' => '150',
            'Default' => '85',
            'Description' => 'Enter Dollar Rate (if you expect USD->local conversion)',
        ),
    );
}

/**
 * Returns the payment button / form shown on invoice page
 */
function e-bdpay_link($params) {
    // If invoice id missing, try to create one automatically
    $invoiceId = isset($params['invoiceid']) ? $params['invoiceid'] : 0;

    if (empty($invoiceId)) {
        // Try to create invoice via localAPI
        $postData = array(
            'userid' => $params['clientdetails']['userid'],
            'sendinvoice' => true,
            'itemdescription1' => 'Invoice for Order #' . ($params['orderid'] ?? '0'),
            'itemamount1' => $params['amount'],
            'itemtaxed1' => false,
        );
        try {
            $create = localAPI('CreateInvoice', $postData);
            if (isset($create['result']) && $create['result'] == 'success' && !empty($create['invoiceid'])) {
                $invoiceId = $create['invoiceid'];
            } else {
                // Failed to create invoice: show fallback message and still attempt to generate payment URL without invoice
                $createError = isset($create['message']) ? $create['message'] : 'Could not auto-create invoice.';
                // We'll show an error message below if needed
            }
        } catch (Exception $e) {
            // ignore and continue: invoice creation failed
        }
    }

    // If user clicked pay button (POST), build payment url and redirect or return form
    if (isset($_POST['pay'])) {
        $response = e-bdpay_payment_url($params, $invoiceId);
        if ($response && !empty($response->status) && $response->status && !empty($response->payment_url)) {
            // Redirect to payment URL
            header('Location: ' . $response->payment_url);
            exit();
        } else {
            // Return message to user
            $message = is_object($response) && isset($response->message) ? $response->message : 'Unable to create payment.';
            return '<div>' . htmlspecialchars($message) . '</div>';
        }
    }

    // Default: show a form with Pay Now button
    $form = '<form action="" method="POST">';
    $form .= '<input class="btn btn-primary" name="pay" type="submit" value="' . $params['langpaynow'] . '" />';
    $form .= '</form>';
    return $form;
}

/**
 * Build payment request to e-bdpay and return decoded response object
 * $params - gateway parameters provided by WHMCS
 * $invoiceId - invoice id to attach to payment (if available)
 */
function e-bdpay_payment_url($params, $invoiceId = null) {
    $cus_name = trim($params['clientdetails']['firstname'] . " " . $params['clientdetails']['lastname']);
    $cus_email = $params['clientdetails']['email'];
    $apikey = $params['apiKey'];
    $currency_rate = isset($params['currency_rate']) ? $params['currency_rate'] : 1;
    $invoiceId = $invoiceId ?: (isset($params['invoiceid']) ? $params['invoiceid'] : 0);
    $amount = $params['amount'];

    // Convert USD to local if configured
    if ($params['currency'] == "USD" && is_numeric($currency_rate) && $currency_rate > 0) {
        $amount = $amount * floatval($currency_rate);
    }

    // Build system base URL
    $urlScheme = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    $systemUrl = $urlScheme . $_SERVER['HTTP_HOST'];

    // Build webhook and redirect URLs
    $webhook_url = $systemUrl . '/modules/gateways/callback/e-bdpay.php?api=' . urlencode($apikey) . '&invoice=' . urlencode($invoiceId);
    $success_url = $systemUrl . '/viewinvoice.php?id=' . urlencode($invoiceId);
    $cancel_url = $systemUrl . '/viewinvoice.php?id=' . urlencode($invoiceId);

    $data = array(
        "cus_name"    => $cus_name,
        "cus_email"   => $cus_email,
        "amount"      => $amount,
        "webhook_url" => $webhook_url,
        "success_url" => $success_url,
        "cancel_url"  => $cancel_url,
    );

    $endpoint = 'https://pay.e-bdpay.com/api/payment/create';
    $headers = array(
        'Content-Type: application/json',
        'API-KEY: ' . $apikey,
    );

    $curl = curl_init();
    curl_setopt_array($curl, array(
        CURLOPT_URL => $endpoint,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($data),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30,
    ));

    $response = curl_exec($curl);
    $curlError = curl_error($curl);
    curl_close($curl);

    if ($response === false || !empty($curlError)) {
        return (object) array('status' => false, 'message' => 'cURL Error: ' . $curlError);
    }

    $res = json_decode($response);
    if ($res === null) {
        return (object) array('status' => false, 'message' => 'Invalid JSON response from payment gateway.');
    }

    return $res;
}
?>